\section{PprUtils}

Automatic instances:
  If sufficient class members are exported
  (and QuickCheck tests of the class are passed),
  generate an instance.

|Instantiate ClassesModule ImplementationModule| $\rightarrow$ 
\begin{code}
module PprUtils where

-- package GHC
import GHC ( TyThing(..)
           , Id, idType
           , DataCon, dataConType, dataConTyCon
           , Class(..), classTvsFds, classMethods
           )
import TyCon (TyCon, tyConKind, tyConName)
import Module   ( ModuleName, mkModuleName, moduleName )
import Name (nameModule_maybe)
import Outputable

import Data.Char (isAlpha)
import System.IO (stderr)

import Debug.Trace
\end{code}

\begin{code}
traceSDoc :: SDoc -> a -> a
traceSDoc sdoc = trace (showSDoc $ withPprStyle defaultDumpStyle sdoc)
\end{code}

\begin{code}
blockComment :: SDoc -> SDoc
blockComment sdoc = text "{-" $+$ sdoc $+$ text "-}"
\end{code}

\begin{code}
pprTyThing :: TyThing -> SDoc
pprTyThing (AnId ident)   = text "Identifier:" <+> pprId ident
pprTyThing (ADataCon dc)  = text "DataCon:   " <+> pprDataCon dc
pprTyThing (ATyCon tc)    = text "TyCon:     " <+> pprTyCon tc
pprTyThing (AClass c)     = text "Class:     " <+> vcat
  (map ppr (classSCTheta c) ++
   (ppr c <+> ppr (classTvsFds c)) :
   map pprId (classMethods c)
  )
\end{code}

\begin{code}
pprId :: Id -> SDoc
pprId ident = pprTyped ident (idType ident)
\end{code}

\begin{code}
pprTyped :: (Outputable a, Outputable ty) => a -> ty -> SDoc
pprTyped x ty = ppr x <+> text "::" <+> ppr ty
\end{code}

\begin{code}
pprDataCon :: DataCon -> SDoc
pprDataCon dc = pprTyped dc (dataConType dc)
\end{code}

\begin{code}
pprTyCon :: TyCon -> SDoc
pprTyCon tc = pprTyped tc (tyConKind tc)
\end{code}

\begin{code}
pprDataConImport :: ModuleName -> DataCon -> SDoc
pprDataConImport wmod wdc = text "import" <+> ppr wmod <+>
              parens (ppr (dataConTyCon wdc) <> parens (ppr wdc))
\end{code}

\begin{code}
pprTyConImport :: ModuleName -> TyCon -> SDoc
pprTyConImport imod tc = text "import" <+> ppr imod <+> parens (ppr tc)
\end{code}

\begin{code}
pprTyConImport' :: TyCon -> SDoc
pprTyConImport' tc = case nameModule_maybe $ tyConName tc of
  Just m   -> pprTyConImport (moduleName m) tc
  Nothing  ->  flip pprTyConImport tc . fst . splitQual
           .   showSDoc . withPprStyle defaultUserStyle $ ppr tc
\end{code}

%{{{ splitQual :: String -> (ModuleName, String)
\begin{code}
splitQual :: String -> (ModuleName, String)
splitQual s = let
  (rident, rest) = break ('.' ==) $ reverse s
  m = case rest of
    [] -> mkModuleName "Prelude"
    _ : rmod -> mkModuleName $ reverse rmod
  in (m, reverse rident)
\end{code}
%}}}

\begin{code}
stringOfId :: Id -> String
stringOfId = showSDocUnqual . ppr
\end{code}

\begin{code}
pprintE :: SDoc -> IO ()
pprintE = printForUser stderr neverQualify
\end{code}

%{{{ pprint :: SDoc -> IO ()
\begin{code}
pprint :: SDoc -> IO ()
pprint sdoc = printSDoc sdoc pprStyle

pprStyle :: PprStyle
pprStyle = mkUserStyle neverQualify (PartWay 80)
\end{code}
%}}}

%{{{ isOperator
|isOperator| is taken from |ghc/compiler/utils/Outputable.lhs|,
where it is not exported.
\begin{code}
isOperator :: SDoc -> Bool
isOperator ppr_v
  = case showSDocUnqual ppr_v of
        ('(':_)   -> False              -- |()|, |(,)| etc
        ('[':_)   -> False              -- |[]|
        ('$':c:_) -> not (isAlpha c)    -- Don't treat |$d| as an operator
        (':':c:_) -> not (isAlpha c)    -- Don't treat |:T| as an operator
        ('_':_)   -> False              -- Not an operator
        (c:_)     -> not (isAlpha c)    -- Starts with non-alpha
        _         -> False
\end{code}
%}}}

\begin{code}
pprHsVarNewQual :: (Outputable a) => String -> a -> SDoc
pprHsVarNewQual modname ident = pprPrefixVar (isOperator pp) (text (modname ++ ".") <> pp)
             where pp = ppr ident
\end{code}

%{{{ EMACS lv
% Local Variables:
% folded-file: t
% fold-internal-margins: 0
% eval: (fold-set-marks "%{{{ " "%}}}")
% eval: (fold-whole-buffer)
% end:
%}}}
